#!/usr/bin/env perl
#
# NOTICE: By January 2024 at the latest this Perl update script will be replaced with a better
#         C++ program that will do things cleaner and in a way that is more robust against
#         problems with core system libraries that would otherwise cause programs not to run.
#         A corresponding change will occur with castus-apply-update.

use strict;
use warnings;

use v5.16;
use feature qw<switch>;

use Config;
use Cwd qw<getcwd abs_path>;
use File::stat;

our $bitness;
our $owndir;
our $is_quickstream;
our $is_recording;
our $enable_lib_static;
our $update;

# sub2ver(<string>)
# ex:
# sub2ver("1.2.3") = ( 1, 2, 3 )
sub str2ver($) {
	my $s = shift @_;
	my @a = split(/\./,$s);
	my @b = ( );
	my $i = 0;

	while ($i < @a) {
		my $e = $a[$i];
		$e =~ s/[^0-9]+$//;
		push(@b,$e+0) if ($e ne "");
		$i++;
	}

	return @b;
}

# ver2str( ( 1, 2, 3 ) ) = "1.2.3"
sub ver2str {
	my $s = "";

	for my $i (@_) {
		$s .= "." if ($s ne "");
		$s .= $i;
	}

	return $s;
}

# vercmp(\@a,\@b)
# vercmp( [1, 2, 3], [1, 2, 4] )
# return -1 if @a < @b
# return 0 if @a == @b
# return 1 if @a > @b
sub vercmp($$) {
	my @a = @{shift @_}; # \@arr
	my @b = @{shift @_}; # \@arr
	my $count = @a;
	$count = @b if $count < @b;
	my $av;
    my $bv;
    my $i;
	for ($i=0;$i < $count;$i++) {
		$av = $bv = 0;
		$av = $a[$i] if $i < @a;
		$bv = $b[$i] if $i < @b;
		return -1 if $av < $bv;
		return 1 if $av > $bv;
	}
	return 0;
}

# $glibc_ver is something like this now:
#
# ldconfig (GNU libc) 2.17
# Copyright (C) 2012 Free Software Foundation, Inc.
# This is free software; see the source for copying conditions.  There is NO
# warranty; not even for MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
# Written by Andreas Jaeger.
#
# We need the version number on the first line.
sub ldconfigver($) {
	my $str = shift @_;
	my $r = undef;

	my @t = split(/\n/,$str);
	if (@t > 0) {
		my $tmp = $t[0];
		my @a = split(/[ \t]+/,$tmp);
		if (@a >= 3) {
			$tmp = $a[3];
			if ($tmp =~ m/^[0-9]+\.[0-9]+$/) {
				$r = $tmp;
			}
		}
	}

	return $r;
}

sub linuxkernelver() {
    my $str;
    my $r = undef;

    if (open(LKV,"<","/proc/version")) {
        $str = <LKV>;
        chomp $str;
        close(LKV);
    }

    # Linux version 4.19.82-isptv (nobody@castus4-compile-1) (gcc version 4.6.3 (GCC)) #1 SMP PREEMPT Sun Aug 16 09:19:53 -00 2020
    if ($str =~ m/^Linux version ([0-9\.]+)/) {
        $r = $1;
    }

    return $r;
}

sub statsize {
    my $path = shift;
    return 0 if !defined($path);
    my $x = stat($path);
    return 0 if !defined($x);
    return $x->size;
}

sub get_bitness {
    given ($Config{archname}) {
        when (/^x86_64-.*/) { return 64; }
        # TODO: Test 32-bit
        when (/^i686-.*/)   { return 32; }
        default { die "Unsupported host"; }
    }
}

sub get_owndir {
    my $desired = abs_path("/mnt/main/tv/install");
    if (not defined $desired) {
        die "Cannot determine main path";
    }

    my $dir = abs_path(getcwd());
    if ($dir ne $desired) {
        die "Must run from ${desired}, not ${dir}";
    }
    if (not -d $dir) {
        die "Install directory is malformed";
    }

    return $dir;
}

sub get_quickstream {
# This test looks for rendermgr/channel# to determine if it is a QuickStream.
# This is a bad test because this script aborts if there is no tv/rendermgr/ directory.
# LAH commented out the fail for that condition and added "return 1;"
#
# Therefore, the software update should be run after the channels are added.
# Note that this function is used to change the chmod attributes to disable autostart
# on QuickRolls and QuickCasts.  When you click on the button,
# it is run in a way that it bypasses the attributes.

    if (opendir(my $dirh, "/mnt/main/tv/rendermgr")) {
        if (+(grep { /^channnel\d+$/ } readdir($dirh)) == 0) {
            return 1;
        } else {
            return 0
        }
        closedir($dirh);
    } else {
        return 1;
    }
}

sub get_lib_static {
    if (-f "/usr/lib/libX11.a" or -f "/usr/lib64/libX11.a") {
        return 1;
    } else {
        return 0;
    }
}

sub get_recording {
    my $rec = system("./check4recordinginputs.pl");
    # Status code 60 - one or more inputs are recording or hung
    if ($rec == 0) {
        return 0;
    } elsif (($rec & 0xFF00) == (60 << 8)) {
        return 1;
    } else {
        return 2;
    }
}

sub check_is_castus4 {
    if (not -x "/usr/bin/c4mi_rendermgr") {
        die "CASTUS 4 software not present";
    }
}

sub check_loader_config {
}

sub check_disk_reserve {
    sub get_free_bytes {
        my $fs = shift;
        return `stat -f -c '%f' $fs` * `stat -f -c '%S' $fs`;
    }
    sub get_free_inodes {
        my $fs = shift;
        return `stat -f -c '%d' $fs`;
    }

    my $main_free = get_free_bytes("/mnt/main/");
    if ($main_free < 2**30) {
        die "Cannot update. Main partition does not have enough disk space. Clear some items from your library to remedy the situation.";
    }

    # Recent systems have a problem where Yarn leaves all these /tmp/yarn--* directories all over the
    # place. Over time they can build up to some 10,000 of these in /tmp and eventually exhaust inodes.
    # Before we check free inodes, clean up this mess.
    if ( -x "./yarn-spam-sucks.pl" ) {
        system("./yarn-spam-sucks.pl");
    }

    # It is possible for there to be enough room for the DATA, but not enough INODES to do the update.
    # This can happen if the /tmp directory is chock full of those God damn /tmp/yarn--* files.
    # The result is no less potentially catastrophic than running out of free space during an update.
    my $system_free_inodes = get_free_inodes("/");
    if ($system_free_inodes < 50000) {
        die "Cannot update. Main partition has way too many files. Please contact technical support."
    }

    my $system_free = get_free_bytes("/");
    if ($system_free < 2**30) {
        my $log_size = statsize("/var/log/messages");
        my $reporting_log_size = statsize("/usr/share/castus/reporting/log");
        if ($system_free - $reporting_log_size < 2**30) {
            if ($system_free - $reporting_log_size - $log_size < 2**30) {
                die "Cannot update. System partition does not have enough disk space."
            }
            system("truncate", "-s", "0", "/usr/share/castus/reporting/log");
        }
        system("truncate", "-s", "0", "/var/log/messages");
    }
}

sub get_tar_metadata {
    my $tar = shift;
    my %meta;
    open(my $tarh, '-|', 'tar', '-xf', $tar, '.castus-update', '--to-stdout');
    while (<$tarh>) {
        chomp;
        if (/^([^=]+)=(.*)$/) {
            $meta{$1} = $2;
    }
    }
    close($tarh);
    return \%meta;
}

sub find_update_tarball {
    my @tarballs;

    if (opendir(my $dirh, getcwd())) {
        while (my $entry = readdir($dirh)) {
            next unless $entry;
            next unless $entry =~ /\.tar$/;
            my $meta = get_tar_metadata($entry);
            next unless exists $meta->{date};
            next unless $meta->{product} eq 'castus4';
            if ($bitness == 64) {
                next unless $meta->{target} eq 'x86_64';
            } else { # TODO: Test 32-bit
                next unless $meta->{target} eq 'i686';
            }

            $meta->{path} = $entry;
            push(@tarballs, $meta);
        }
        closedir($dirh);
    } else {
        die "Unable to search for update package";
    }

    @tarballs = sort { $a->date cmp $b->date } @tarballs;
    if (+@tarballs == 0) {
        die "No suitable update tarball found";
    }
    return pop(@tarballs)
}

sub check_hash {
    my $file = shift;
    if (system("sha1sum", "-c", $file . ".sha1") != 0) {
        die "Update integrity check failed; try re-downloading";
    }
}

sub preflight_checks {
    $bitness = get_bitness();
    $owndir = get_owndir();
    $is_quickstream = get_quickstream();
    $enable_lib_static = get_lib_static();
    $is_recording = get_recording();
    check_is_castus4();
    check_disk_reserve();
    $update = find_update_tarball();
    check_hash($update->{path});
}

sub user_confirmation {
    say "Update package: $update->{path} (build $update->{date}) for $bitness-bit systems";
    print "Ready to install? (type Y or yes to confirm): "; $|++;
    my $ready = <STDIN>; chomp $ready;
    if ($ready !~ /^[yY]/) {
        say "Update aborted; exiting";
        exit 0;
    }

    if ($is_recording == 1) {
        say "WARNING: One or more inputs are RECORDING or HUNG at this time.";
        say "If you apply the update, the recordings may be interrupted and";
        say "cut in half.";
        print "Are you sure you want to update? (type Y or yes to confirm): "; $|++;
        my $really = <STDIN>; chomp $really;
        if ($really !~ /^[yY]/) {
            say "Update aborted; exiting";
            exit 1;
        }
    } elsif ($is_recording == 2) {
        say "Unable to determine if anything on the system is interruptable.";
        print "Are you sure you want to update? (type Y or yes to confirm): "; $|++;
        my $really = <STDIN>; chomp $really;
        if ($really !~ /^[yY]/) {
            say "Update aborted; exiting";
            exit 1;
        }
    }

    say "Beginning update"
}

sub run_scripts {
    system("/usr/share/castus4/sql/castus-reporting-init >/dev/null 2>&1");
    system("./control-api-init.sh >/dev/null 2>&1");
}

sub sanitize_symlinks {
    for my $libdir (qw</lib /usr/lib>) {
        if (-l $libdir) {
            unlink($libdir);
        }
    }
}

sub purge_c4lic {
    unlink("/c4lic");
    unlink("/c4lic.x86");
    unlink("/c4lic.x64");
}

# some of the latest exploits on our systems like to put replacements in AND make them immutable!
sub purge_immutables {
    my %immutables = (
        "/bin/" => [qw<ps login>],
        "/usr/bin/" => [qw<ssh sshd tar bash cp mv rm ps top chattr lsattr>],
        "/sbin/" => ['agetty'],
        "/usr/sbin/" => ['sshd'],
        "/usr/sshd-static/sbin/" => [qw<ssh sshd>],
    );
    for my $dir (keys(%immutables)) {
        for my $exec (@{$immutables{$dir}}) {
            # if a file, and not a symbolic link (so that chattr does not show Operation Not supported)
            if ( -f ($dir . $exec)) {
                if ( !( -l ($dir . $exec)) ) {
                    system('chattr', '-aiu', $dir . $exec);
                }
            }
        }
    }
}

sub purge_video_drivers {
    for my $driver (qw<vesa nv>) {
        for my $libdir (qw<lib lib64>) {
            for my $suffix (qw<la so>) {
                unlink("/usr/${libdir}/xorg/modules/drivers/${driver}_drv.${suffix}");
            }
        }
    }
}

sub purge_gcc_4_9 {
    system("rm", "-rf", "/usr/gcc-4.9");
}

sub purge_ffmpeg_4 {
    system("rm", "-rf", "/usr/ffmpeg-4");
}

sub purge_libfbdevhw {
    for my $libdir (qw<lib lib64>) {
        for my $suffix (qw<a la so>) {
            unlink("/usr/${libdir}/xorg/modules/linux/libfbdevhw.${suffix}");
        }
    }
}

sub purge_egg_info_dirs {
    my $base = "/usr/lib";
    if (opendir(EI,$base)) {
        while (my $rd = readdir(EI)) {
            next unless $rd =~ m/^python/;
            my $path = "$base/$rd";
            next unless -d $path;
            system("find", $path, "-name", "*.egg-info", "-exec", "rm", "-rf", "{}", "+");
        }
        closedir(EI);
    }
}

sub sanitize_ld_config {
    my $x = -1;
    my $y = -1;
    my $z = -1;
    my @a = qw();
    if (open(X,"/etc/ld.so.conf")) {
        $x = 0;
        $y = 0;
        $z = 0;
        while (!eof(X)) {
            my $line = <X>;
            chomp $line;

            if ($line eq "/usr/gcc-4.8/lib") {
                $x = 1;
            }
            elsif ($line eq "/usr/gcc-4.8/lib64") {
                $x = 1;
            }
            elsif ($line eq "/usr/gcc-9.3/lib") {
                $y = 1;
            }
            elsif ($line eq "/usr/gcc-9.3/lib64") {
                $y = 1;
            }
            else {
                push(@a,$line);
            }
        }
        close(X);

        if ($x == 0 || $y == 0 || $z == 0) {
            print "Updating ld.so.conf\n";

            if (open(Y,">/etc/ld.so.conf.new")) {
                print Y "/usr/gcc-9.3/lib64\n";
                print Y "/usr/gcc-9.3/lib\n";
                print Y "/usr/gcc-4.8/lib64\n";
                print Y "/usr/gcc-4.8/lib\n";
                for (my $i=0;$i < @a;$i++) {
                    print Y $a[$i]."\n";
                }
                close(Y);
                rename("/etc/ld.so.conf.new","/etc/ld.so.conf");
                system('sync >/dev/null 2>&1');
                system("ldconfig >/dev/null 2>&1");
                system('sync >/dev/null 2>&1');
            }
        }
    }
}

sub ensure_depmod {
    system("depmod", "-a", "-m");
}

sub purge_prev_samba {
    system("rm -Rf /usr/{lib,lib64}/samba");
}

sub purge_firefox16 {
    system("rm -Rf /usr/lib/firefox-16.0.2 /usr/bin/firefox16 /usr/share/applications/firefox16.desktop");
}

sub purge_firefox21 {
    system("rm -Rf /usr/lib/firefox-21.0 /usr/bin/firefox21 /usr/share/applications/firefox21.desktop");
}

sub purge_firefox40 {
    system("rm -Rf /usr/lib/firefox-40.0 /usr/bin/firefox40 /usr/share/applications/firefox40.desktop");
}

sub purge_node_for_reinstall {
    # why the hell did Node 22 decide to put itself into /bin??
    system("rm -Rf /lib/node_modules /usr/lib/node_modules");
    system("rm -f /bin/node /usr/bin/node /bin/yarn /usr/bin/yarn /bin/yarnpkg /usr/bin/yarnpkg /bin/npm /usr/bin/npm /bin/npx /usr/bin/npx");
}

# Old Castus installs up until 2017 had a libgiofam.so module that, once the system is updated,
# breaks ABI and causes most things on the desktop to crash. Remove it.
sub purge_old_libgiofam {
    system("rm -f /usr/{lib,lib64}/gio/modules/libgiofam.{la,so}");
}

sub cleanup_prepare {
    sanitize_symlinks();
    purge_c4lic();
    purge_immutables();
    purge_video_drivers();
    purge_gcc_4_9();
    purge_ffmpeg_4();
    purge_libfbdevhw();
    purge_egg_info_dirs();
    purge_prev_samba();
    purge_firefox16();
    purge_firefox21();
    purge_firefox40();
    purge_node_for_reinstall();
    purge_old_libgiofam();
    sanitize_ld_config();
    ensure_depmod();
}

sub tar_retryloop {
    my $sub = $_[0];
    my $rs = undef;
    my $ret = 0;

    if (defined($sub)) {
        while (1) {
            $ret = $sub->();
            last if ($ret == 0);

            print "\n";
            while (1) {
                print "A problem occurred. Answer and hit enter: (I)gnore (R)etry (A)bort? "; $|++;
                $rs = lc(<STDIN>); chomp $rs;
                if ($rs eq 'r' || $rs eq 'retry') {
                    $rs = 'r';
                    last;
                }
                elsif ($rs eq 'a' || $rs eq 'abort') {
                    $rs = 'a';
                    last;
                }
                elsif ($rs eq 'i' || $rs eq 'ignore') {
                    $rs = 'i';
                    last;
                }
            }

            print "\x0D"."\x1B[K"."\x0D"; $|++; # erase line

            if ($rs eq 'r') {
                next; # try again
            }
            elsif ($rs eq 'i') {
                last; # NTS: the calling code currently ignores return value
            }
            elsif ($rs eq 'a') {
                $ret = 1;
                print "Update aborted\n";
                exit 1;
            }
        }
    }

    return $ret;
}

# ldconfig worst case scenario can segfault if shared objects are really fucked up.
# This detects that and changes to zero so that ldconfig in that case doesn't require
# the user to constantly hit "ignore".
sub segfault_is_ok($) {
    my $ret = $_[0];

    $ret = 0 if ($ret & 0xFF00) == 0x8b00; # SIGSEGV = 11

    return $ret;
}

sub install_packages {
    my @criticals = (
        'busybox',
        'backup-glibc',
        'locale-gen',
        $bitness == 64 ? ('glibc-32') : (),
        'glibc',
        'libcap',
        'attr',
        'xz-utils',
        'gcc',
        'tar',
        'perl',
        'Linux-PAM',
        'util-linux',
        'ncursesw',
        'zlib',
        'openssl',
        'openssh'
    );

    system("tar", '-xf', $update->{path}, "binary${bitness}");

    my $cur = 1;
    my @all = ();
    if (opendir(my $dirh, "binary${bitness}")) {
        while (my $entry = readdir($dirh)) {
            next unless -f "binary${bitness}/$entry";
            next unless $entry =~ /\.tar\.xz$/;
            push(@all, $entry);
        }
        closedir($dirh);
        @all = sort(@all);
    } else {
        die "Unable to find binaries";
    }

    # we have a stripped down GNU tar we can use to continue unpacking even during the criticals when it might stop working
    if ( -f "./tar.installer" ) {
        system("chmod 0755 ./tar.installer");
    }
    else {
        die "Missing tar.installer file";
    }

    my $ret;
    my $tarcmd = "tar";

    for my $critical (@criticals) {
        printf("%d/%d Extracting package ${critical}.tar.xz", $cur, scalar @all);
        $ret = tar_retryloop( sub { return system($tarcmd, "--warning=none", '-C', '/', '-xJf', "binary${bitness}/${critical}.tar.xz"); } );
        unlink("binary${bitness}/${critical}.tar.xz");
        $ret = tar_retryloop( sub { return segfault_is_ok(system('ldconfig >/dev/null 2>&1')); } );
        if ($critical eq "glibc") {
            # locale-gen. It will say something so clear the line and let it yammer.
            print "\x0D"."\x1B[K"."\x0D"; $|++; # erase line
            system('locale-gen');
            # we can now use our own tar
            $tarcmd = "./tar.installer";
        }
        if ($critical eq "tar") {
            # by this point the new tar's dependencies have been installed (attr+libcap) we can switch back now
            $tarcmd = "tar";
        }
        if ($critical eq "locale-gen") {
            if ( -f "/etc/locale.gen.example" ) {
                system("cp -n /etc/locale.gen.example /etc/locale.gen");
            }
        }
        print "\x0D"."\x1B[K"."\x0D"; $|++; # erase line
        $cur++;
    }

    for (@all) {
        next unless -f "binary${bitness}/$_";
        next unless /\.tar\.xz$/;

        if (/^php-with-apache-redirect/) {
            next if ( -f "/usr/apache/htdocs/index.php" );
        }

        printf("%d/%d Extracting package $_", $cur, scalar @all);
        if (/^firefox/ || /^youtube-dl/) {
            $ret = tar_retryloop( sub { return system("tar", "--warning=none", '-C', '/', '--keep-newer-files', '--warning=no-ignore-newer', '-xJf', "binary${bitness}/$_"); } );
        } else {
            $ret = tar_retryloop( sub { return system("tar", "--warning=none", '-C', '/', '-xJf', "binary${bitness}/$_"); } );
        }

        unlink("binary${bitness}/$_");
        $ret = tar_retryloop( sub { return segfault_is_ok(system('ldconfig >/dev/null 2>&1')); } );
        print "\x0D"."\x1B[K"."\x0D"; $|++; # erase line
        $cur++;
    }

	$cur = 1;
	@all = ();
	if (opendir(my $dirh, "binary${bitness}")) {
		while (my $entry = readdir($dirh)) {
			next unless -f "binary${bitness}/$entry";
			next unless $entry =~ /\.deb$/;
			push(@all, $entry);
		}
		closedir($dirh);
		@all = sort(@all);
	} else {
		die "Unable to find binaries";
	}

	for (@all) {
		next unless -f "binary${bitness}/$_";
		next unless /\.deb$/;

		printf("%d/%d Installing package $_", $cur, scalar @all);
		system("dpkg -i binary${bitness}/$_ >/dev/null"); # dpkg has no "silent" mode, so direct output to /dev/null

		unlink("binary${bitness}/$_");
		$ret = tar_retryloop( sub { return segfault_is_ok(system('ldconfig >/dev/null 2>&1')); } );
		print "\x0D"."\x1B[K"."\x0D"; $|++; # erase line
		$cur++;
	}

	unlink("binary${bitness}");
}

sub purge_udev_raid_rules {
    unlink("/usr/libexec/udev/rules.d/64-md-raid.rules");
    unlink("/usr/libexec/udev/rules.d/63-md-raid-arrays.rules");
    unlink("/etc/udev/rules.d/udev-md-raid.rules");
}

sub create_users {
    for (qw<avahi pulse messagebus>) {
        system("useradd -U -M $_ >/dev/null 2>&1");
    }
}

sub fix_vsftpd_conf {
    system("./vsftpd-ipv4-ipv6-fix.pl 2>/dev/null");
}

sub enable_http_conf_d {
    system("./apache-httpd-conf-patch.pl");
    system("./apache-httpd-conf-php-patch.pl");
    system("./apache-httpd-ssl-conf-patch.pl");
    system("rm -f /usr/apache/conf.d/service-control-api.ssl.conf"); # <- that was a mistake
}

sub purge_pulse_consolekit {
    for my $libdir (qw<lib lib64>) {
        unlink("/usr/${libdir}/pulse-8.0/modules/module-console-kit.so");
    }
}

sub purge_license_code {
    unlink("/usr/include/isp-utils-v4/media-item/enc1.h");
    unlink("/usr/include/isp-utils-v4/media-item/lic.h");
}

sub purge_lafiles {
    system("find", "/usr/lib", "/usr/lib64", "-type", "f", "-name", "libisp*.la", "-delete");
}

sub update_font_cache {
    system("fc-cache >/dev/null 2>&1");
}

sub reload_udev {
    system("udevadm", "control", "--reload-rules");
    system("udevadm", "trigger");
}

sub ensure_ssh_root {
    system("/usr/share/castus4/sshd-root-login-default");
}

sub ssh_security_fixes {
    system("/usr/share/castus4/sshd-terrapin-fix");
}

sub fix_postgres_data_dir {
    system("chmod 0700 /mnt/main/tv/postgres/data >/dev/null");
}

sub fix_var_empty_perms {
    chmod(0755, "/var/empty");
}

sub restart_sshd {
    say "Restarting SSH daemon; should not affect existing connections";
    if (system("/usr/share/castus4/restart-sshd") != 0) {
        say "WARNING: error restarting SSH daemon";
    }
}

sub restart_apache {
    # stop Apache
    system("/etc/init.d/service-apache", "stop");
    system("killall -w httpd"); # make SURE it shuts down!
    system("/etc/init.d/service-apache", "stop");
    system("killall -w httpd"); # make SURE it shuts down!
    # then start it again
    system("/etc/init.d/service-apache", "start");
}

sub purge_gcc_4_8 {
    # updating to gcc-4.8.5 as the main compiler means the alt compiler is no longer needed
    system("rm", "-rf", "/usr/gcc-4.8");
}

sub purge_gcc_9_3 {
    # updating to gcc-9.3.0 as the main compiler means the alt compiler is no longer needed
    system("rm", "-rf", "/usr/gcc-9.3");
}

sub purge_sshd_static {
    # static openssh will break with new GLIBC
    if (-d "/usr/sshd-static") {
        system("chattr", "-R", "-i", "/usr/sshd-static");
        system("rm", "-rf", "/usr/sshd-static");
    }
}

sub purge_usr_share_doc {
    # /usr/share/doc contains almost 400MB of stuff nobody reads (at least our non-technical users!)
    # some clients are close to running out of space on the system partition, so this helps.
    system("rm", "-rf", "/usr/share/doc");
}

sub immutable_ssh {
    # some of the latest exploits on our systems like to put replacements in AND make them immutable!
    # WE make them immutable first!
    system("chattr", "+i", qw</usr/bin/ssh /usr/sbin/sshd>);
}

sub restart_crtv_monitor {
    system("/etc/init.d/service-castus-crtv", "stop");
    sleep(5);
    system("/etc/init.d/service-castus-crtv", "stop");
}

sub fix_perms {
    chmod(0755, "/", "/usr", "/usr/bin", "/usr/sbin");
    chmod(0755, "/root/.nvm/versions/node/v10.0.0/bin"); # < What????
}

sub twiddle_vsftpd {
    # FTP security was a big problem on QuickRolls and QuickCasts in the past
    # so disable FTP on update and power up for QuickRolls and QuickCasts only.
    # If a QuickStream (no channels), enable on update/power up
    # Note that when you click the start button,
    # it bypasses the attribute so it still starts.
    if ($is_quickstream) {
        chmod(0755, "/etc/init.d/service-vsftpd");
    } else {
        chmod(0644, "/etc/init.d/service-vsftpd");
    }
}

sub purge_nscd {
    system("killall", "nscd");
    system("rm", "-rf", "/var/db/nscd");
}

sub compile_glib_schemas {
    # Firefox 55 will crash on "save as" unless glib's schemas are compiled
    system("glib-compile-schemas /usr/share/glib-2.0/schemas >/dev/null 2>&1");
}

sub install_filter_scripts {
    # update installed filter scripts
    system("(echo upgrade; yes '') | (timeout 10 /usr/bin/castus-example-filter-scripts >/dev/null 2>&1)");
}

sub update_v5_web_interface {
    my $flagfile = "/usr/apache/htdocs/.auto-changed-v5";

    # update to Castus v5
    if ( !( -e $flagfile ) ) {
        if ( -x "/usr/bin/change-castus-interface" ) {
            if (system("change-castus-interface 5") == 0) {
                system("echo >$flagfile");
            }
        }
    }

    # apparently this is required now?
    system("mkdir -p /mnt/main/tv/logs/log");
}

sub gcc_cc_symlink {
    if ( !( -e "/usr/bin/cc" ) && ( -x "/usr/bin/gcc") ) {
        symlink("gcc","/usr/bin/cc");
    }
    if ( !( -e "/usr/bin/c++" ) && ( -x "/usr/bin/g++") ) {
        symlink("g++","/usr/bin/c++");
    }
}

sub remove_obsolete_blackmagic_decklink_commands {
    # THIS IS CAUSING PROBLEMS, DISABLE!
    # This code needs to detect if 12.x drivers are installed, and only
    # do this step if 12.x or higher are installed!
    return;

    # Black Magic Design 12.x drivers remove the BlackmagicFirmwareUpdate
    # command entirely. The old command will show the cards as "up to date"
    # when in fact they are NOT, and it cannot update firmware. The new
    # commands is the incredibly descriptive title of DesktopVideoUpdateTool.
    system("rm -Rf /usr/{lib,lib64}/blackmagic/DesktopVideo/{AdminUtility,BlackmagicFirmwareUpdater,BlackmagicFirmwareUpdaterGui,BlackmagicPreferencesStartup}");
    system("rm -Rf /usr/bin/{BlackmagicFirmwareUpdater,BlackmagicFirmwareUpdaterGui,BlackmagicPreferencesStartup}");
}

sub purge_watchlist {
    system("daemon --noconfig --name=castus.watchlist.api --stop >/dev/null 2>&1");
    system("rm -Rf /usr/share/castus/watchlist");
}

sub purge_python2 {
    # Python 2 is suddenly deadly radioactive nuclear waste to paranoid IT departments.
    # All development is Python 3 these days anyway.
    system("rm -Rf /usr/lib/python2.7 /usr/bin/python /usr/bin/python2.7 /usr/bin/python2 /usr/bin/python-config /usr/bin/python2-config");
}

sub purge_older_python {
    # We updated from Python 3.8 to Python 3.10. Keep disk space down by removing the older one.
    system("rm -Rf /usr/lib/python3.8 /usr/bin/python3.8{,-config}");
}

sub clear_python_cache {
    # clear __pycache__ directories. Current version is Python 3.10
    my $pydir = "/usr/lib/python3.10";
    if ( -d $pydir ) {
        die if $pydir eq ""; # don't cd to "", that's the user's home directory!
        system("cd $pydir && find -name __pycache__ -type d -exec rm -Rf {} +");
    }
}

sub purge_old_ruby_gems {
    # current version 2.7.0, purge old Ruby gems from older versions no longer in use
    system("rm -Rf /usr/{lib,lib64}/ruby/{1.9.1,2.2.0}");
    system("rm -Rf /usr/{lib,lib64}/ruby/gems/{1.9.1,2.2.0}");
}

sub purge_old_ps {
    # old systems installed an old /bin/ps that now complains about the kernel HZ value, remove it.
    # but only if /usr/bin/ps exists
    if ( -x "/usr/bin/ps" && -x "/bin/ps" ) {
        unlink("/bin/ps");
        symlink("/usr/bin/ps","/bin/ps");
    }
}

sub dev_shm_security_fix {
    # security fix: do not allow suid-exec, executable, or device nodes in /dev/shm
    system("mount -o rw,remount,nosuid,nodev,noexec /dev/shm >/dev/null 2>&1");
}

sub tmp_security_fix {
    # security fix: /tmp sticky bit so user IDs cannot mess with other user IDs tmp files
    system("chmod ugo+t /tmp >/dev/null 2>&1");
    system("chmod ugo+rwx /tmp >/dev/null 2>&1");
}

sub purge_excess_apache {
    system("rm -Rf /usr/apache/man /usr/apache/share/man /usr/apache/share/doc");
}

sub cleanup_after {
    gcc_cc_symlink();
    purge_old_ps();
    purge_python2();
    purge_older_python();
    purge_old_ruby_gems();
    purge_udev_raid_rules();
    purge_pulse_consolekit();
    purge_license_code();
    clear_python_cache();
    purge_lafiles();
    purge_gcc_4_8();
    purge_gcc_9_3();
    purge_sshd_static();
    purge_usr_share_doc();
    fix_perms();
    create_users();
    fix_vsftpd_conf();
    enable_http_conf_d();
    update_font_cache();
    reload_udev();
    ensure_ssh_root();
    ssh_security_fixes();
    fix_var_empty_perms();
    fix_postgres_data_dir();
    immutable_ssh();
    restart_sshd();
    restart_apache();
    restart_crtv_monitor();
    twiddle_vsftpd();
    compile_glib_schemas();
    install_filter_scripts();
    update_v5_web_interface();
    remove_obsolete_blackmagic_decklink_commands();
    dev_shm_security_fix();
    tmp_security_fix();
    purge_watchlist();
    purge_excess_apache();
}

sub suppress_services {
    my $services = shift;
    my $task = shift;

    for my $service (@{$services}) {
        say "Stopping ${service}";
        system("/etc/init.d/${service} stop >/dev/null 2>&1");
    }

    $@ = undef;
    eval {
        $task->();
    };
    my $err = $@;

    for my $service (reverse @{$services}) {
        say "Starting ${service}";
        system("/etc/init.d/${service} start >/dev/null 2>&1");
    }

    die $err if $err;
}

sub offer_full_restart {
    {
        say "If the restart process takes too long (more than 60 seconds), then a component is";
        say "hung and you should restart the system with CTRL+ALT+DELETE or by power cycling.";
        say "this unit.";
        say "";
        say "Restarting castus software... please wait... "; $|++;
        #----
        system("/etc/init.d/service-hls-relay stop >/dev/null 2>&1");
        system("/etc/init.d/service-control-api stop >/dev/null 2>&1");
        system("c4mi_chctl -c stop -o all 2>&1");
        system("c4mi_chctl -c stop -o autostart 2>&1");
        system("c4mi_inctl -c stop -o all 2>&1");
        system("c4mi_inctl -c stop -o autostart 2>&1");
        system("c4mi_outctl -c stop -o all 2>&1");
        system("c4mi_outctl -c stop -o autostart 2>&1");
        system("c4mi_routectl -c stop -o all 2>&1");
        system("c4mi_routectl -c stop -o autostart 2>&1");
        #----
        system("(echo 65536 >/sys/fs/cgroup/castus_media_item/cpu.shares) 2>/dev/null"); # just in case rmdir fails prioritize media items
        system("rmdir /sys/fs/cgroup/castus_media_item/idle >/dev/null 2>&1");
        system("rmdir /sys/fs/cgroup/castus_media_item >/dev/null 2>&1");
        system("rmdir /sys/fs/cgroup/castus_output/idle >/dev/null 2>&1");
        system("rmdir /sys/fs/cgroup/castus_output >/dev/null 2>&1");
        system("rmdir /sys/fs/cgroup/castus_rendermgr/idle >/dev/null 2>&1");
        system("rmdir /sys/fs/cgroup/castus_rendermgr >/dev/null 2>&1");
        #----
        system("timeout 10 killall-c4mi-update >/dev/null 2>&1");
        system("timeout 10 killall -w -KILL c4mi_grab ctl.cgi c4mi_chctl c4mi_outctl c4mi_inctl c4mi_routectl c4mi_stillimage c4_timer_record_d >/dev/null 2>&1"); sleep 2;
        system("timeout 10 killall -w -KILL c4_timer_record_d >/dev/null 2>&1"); sleep 2;
        say "done.";
    }
    system("/etc/init.d/service-castus4-timer-record start >/dev/null 2>&1");
    system("/etc/init.d/service-hls-relay start >/dev/null 2>&1");
    system("/etc/init.d/service-control-api start >/dev/null 2>&1");
    system("/etc/init.d/services start >/dev/null 2>&1"); sleep 2;
    system("/etc/init.d/services start >/dev/null 2>&1"); sleep 2;
    system("/etc/init.d/services start >/dev/null 2>&1");
    system("/etc/init.d/service-castus4-timer-record start >/dev/null 2>&1");
    say "done.";
}

# why were these installed??
sub remove_reporting_temp {
    system("rm -f /etc/init.d/service-castus4-reporting-public-temp");
    system("rm -f /etc/init.d/service-reporting-rejoin-temp");
    system("rm -f /etc/init.d/service-reporting-ingest-temp");
}

sub remove_watchlist_api {
    my $script = "/etc/init.d/service-watchlist-api";
    # stop the service, remove init.d script
    if ( -f $script ) {
        system("$script stop >/dev/null 2>&1");
        system("rm -f $script");
    }
    # remove the watchlist installation
    system("rm -Rf /usr/share/castus/watchlist");
}

sub remove_reporting {
    system("rm -Rf /usr/share/castus/reporting 2>/dev/null");
}

sub remove_unused_daemons {
    # TSA made us remove these. We've never used these anyway. As if Castus will ever allow you to Telnet into it!
    system("rm -f /usr/libexec/{rshd,tftpd,rlogind,talkd,ftpd,uucpd,rexecd,telnetd} >/dev/null 2>&1");
    # Unfinished reporting service
    system("killall -KILL -w c4_reporter >/dev/null 2>&1");
    system("rm -f /etc/init.d/service-castus4-reporting2 /usr/bin/c4_reporter >/dev/null 2>&1");
}

sub fix_permissions {
    system("chmod 0755 /bin /usr/bin /sbin /usr/sbin /usr/lib /usr/lib64 /usr/libexec /lib /lib64 /etc /usr /var /boot /mnt >/dev/null 2>&1");
    system("chmod 0644 /etc/passwd /etc/group /etc/shadow >/dev/null 2>&1");
    system("chmod 0700 /root >/dev/null 2>&1");
}

sub remove_control_api {
    system("rm -Rf /usr/share/castus/control-api 2>/dev/null");
    system("rm -Rf /usr/share/castus/caption-synchronizer 2>/dev/null");
}

sub write_suffix_version() {
    if (exists($update->{suffix})) {
        if (open(SF,">","/usr/share/castus4/versions/update-package-suffix.nfo")) {
            print SF ($update->{suffix})."\n";
            close(SF);
        }
    }
}

sub main {
    preflight_checks();
    user_confirmation();
    system("mkdir -p /usr/etc/cron.d >/dev/null 2>&1");
    suppress_services(
        [
            'service-vsftpd',
            'service-castus4-index',
            'service-castus4-rtmp-service',
            'service-reporting-rejoin-temp',
            'service-reporting-rejoin',
            'service-reporting-ingest-temp',
            'service-reporting-ingest',
            'service-caption-synchronizer',
            'service-castus4-reporting-public-temp',
            'service-castus4-reporting-public',
            'service-castus4-reporting',
            'service-embedded-schedule',
            'service-control-api',
            'service-dbus-then-avahi',
            'service-dbus',
            'service-syslog'
        ],
        sub {
            run_scripts();
            remove_reporting();
            remove_reporting_temp();
            remove_watchlist_api();
            remove_control_api();
            remove_unused_daemons();
            fix_permissions();
            suppress_services(
                [
                    'service-postgres'
                ],
                sub {
                    cleanup_prepare();
                    install_packages();
                    write_suffix_version();
                    cleanup_after();
                    fix_permissions();
                }
            );
        }
    );
    system("rm -f ./control-api-init.sh"); # castus-apply-update does not remove this
    offer_full_restart();
    # use the shell to suck up all terminal IO pending
    system("while read -t 1 X; do true; done");
}

# minimum kernel version check
my $linuxkernelstr = linuxkernelver();
my @linuxkernelver = str2ver($linuxkernelstr);
my @kernelmin = ( 4, 1, 20 );

if (vercmp(\@linuxkernelver,\@kernelmin) < 0) {
    print "Your system is running Linux kernel ".ver2str(@linuxkernelver).", which is too old.\n";
    print "This update works best with at least Linux kernel ".ver2str(@kernelmin).".\n";
    print "Type YES and hit enter to continue, or hit enter to cancel.\n";
    my $x = <STDIN>;
    chomp $x;
    exit 0 unless lc($x) eq "yes";
}

main(@ARGV);
